function m = mfs_mce(elem)

# usage: m = mfs_mce(elem)
#
# Input  elem     Structure containing data of one element
#
# Output m        Element mass matrix
#
# The function computes the consistent mass matrix of a 3-dimensional
# trianguler membrane element with 3 nodes with drilling dofs.
#
# -------------------------------------------------------------------

  persistent ng; % Gauss integration
  persistent r; persistent  s; persistent w;

  persistent ib; % Indices of edge end nodes
  persistent iu; persistent iv; persistent iw; % Dof indices
  persistent iuvr;
  persistent T63;

# Initialize

  if (isempty(ng))
     ng = 3;               % cf. Bathe
     f  = 1 / 6;
     r  = [0.5, 0.5, 0];
     s  = [0, 0.5, 0.5];
     w  = f * [1, 1, 1];
     ib = [2, 3, 1];
     iu = [1 : 2 : 11]; iv = iu + 1;                   % refers to t6
     iw = [3, 7, 11];                                  % refers to t3r
     iuvr = [1, 2, 4, 5, 6, 8, 9, 10, 12];             % refers to t3r
     T63 = [speye(3, 3); 
            0.5, 0.5, 0; 0, 0.5, 0.5; 0.5, 0, 0.5];    % t6 -> t3
  end

# Mass density

  if (isfield(elem.mat, "rho"))
     rho = elem.mat.rho;
  else
     m = sparse(18, 18);
     return;
  end

# Transformation to element coordinate system

  [coor, TE] = mfs_esys(elem.coor, 2);

# Transformation to t6 element

  TR =  mfs_rot2mid2d(coor);

# Add coordinates of midpoints

  coor = [coor; 0.5 * (coor + coor(ib, :))];

# Mass matrix of q8 element

  m66 = zeros(6, 6, "double");

  for n = 1 : ng
      JE = mfs_JE_t6(r(n), s(n), coor(:, 1), coor(:, 2));
      wdA = w(n) * JE;
      N   = mfs_NE_t6(r(n), s(n));
      m66 += N' * N * wdA;
  end

  m66 = rho * elem.geom.t * m66;

  mt6         = sparse(12, 12);
  mt6(iu, iu) = m66; mt6(iv, iv) = m66;

# Mass matrix of 2-dim. t3r element

  m2dt3r = TR' * mt6 * TR;

# Mass matrix of 3-dim. t3r element in local coordinate system

  ml  = zeros(12, 12, "double"); 
  m33 = T63' * m66 * T63;

  ml(iuvr, iuvr) = m2dt3r;
  ml(iw, iw)     = m33;

# Transformation to global coordinate system

  m = TE' * ml * TE;

end
