function [ixdof, rc, noddof] = mfs_finddof(nodes, mxdofpnt, nodeid, dof, nset);

# usage: [ixdof, rc, noddof] = mfs_finddof(nodes, mxdofpnt, nodeid, dof, nset)
#
# Input  nodes(:)     List of all nodal point identifiers
#        mxdofpnt     Maximum number of dofs per node
#        nodeid(:)    List of nodal point identifiers to be found
#                     or cell array with nodal point set names
#        dof(:)       List of degree of freedom identifiers
#        nset         Structure with definition of nodal point sets
#
# Output ixdof(:)     Indices of dofs in user dof matrices
#        rc           Return code: 0 means no errors
#        noddof(:, 2) 1st col: nodal point indices
#                     2nd col: degree of freedom identifiers
#
# ---------------------------------------------------------------------

  ixdof = []; noddof = [];
  rc    = 0;

  if (iscell(nodeid))

# Response is defined by set names and dofs

     dofs = cell2mat(dof);

     bad1 = find(dofs < 1); bad2 = find(dofs > mxdofpnt);
     baddof = union(bad1, bad2);

     if (! isempty(baddof))
        rc = 1;
        for ix = baddof
            printf("*E* bad dof identifier %2d for node set %s\n",
                   dofs(ix), nodeid{ix});
        end
     end

     nodix = []; dof = [];

     for n = 1 : length(nodeid)
         sname = nodeid{n};
         if (! ischar(sname))
            rc = 1;
            printf("*E* bad definition of response identifiers\n");
         elseif (isfield(nset, sname))
            newids = nset.(sname);
            ln     = length(newids);
            nodix  = [nodix, newids];
            dof    = [dof, dofs(n)(ones(1, ln))];
         else
            rc = 1;
            printf("*E* nodal point set %s does not exist\n",
                   sname);
         end
     end

  else

# Response is defined by nodal point identifiers and dofs

     nodeid = nodeid(:)';
     dof    = dof(:)';

     bad1 = find(dof < 1); bad2 = find(dof > mxdofpnt);
     baddof = union(bad1, bad2);

     if (! isempty(baddof))
        rc = 1;
        for ix = baddof
            printf("*E* bad dof identifier %2d for node %6d\n",
                   dof(ix), nodeid(ix));
        end
     end

     nodix = lookup (nodes, nodeid, "m");

     badnod = find(nodix == 0);

     if (! isempty(badnod))
        rc = 1;
        for ix = badnod
            printf("*E* node %6d does not exist\n", nodeid(ix));
        end
     end

  end

  if (rc) return; end

  ixdof = (nodix - 1) * mxdofpnt + dof;

  noddof(:, 1) = nodix';
  noddof(:, 2) = dof';

end
