function [cmp, f, ES, WR, rc] = mfs_freqrespx(cmp, f, args)

# usage: [cmp, f, ES, WR, rc] = mfs_freqrespx(cmp, f, args)
#
# Input  cmp        Structure with component
#        f(:)       List of excitation frequencies
#        args       Parameters
#
# Output cmp        Structure with component
#                   (Frequency response added)
#        f(:)       List of excitation frequencies (optional)
#        ES(:)      Strain energies
#        WR(:)      Work of residual
#        rc         Return code: 0 = no errors
#                                1 = errors
#
# Parameters:
#
#   Name       Values    Default   Meaning
#  -----------------------------------------------------------------------
#   method     enhanced  enhanced  Enhanced modal reduction (default)
#              modal               Modal reduction
#              direct              Direct frequency response
#   nband       int        5       Excitation frequencies per
#                                  halfpower bandwidth
#   loadcase    int        1       Loadcase number
#
# ------------------------------------------------------------------------

# Copyright (c) 2017 by Johannes Wandinger

# Check arguments

  if (nargin != 3 || nargout != 5)
     print_usage();
  end

# Initialize output

  ES = 0;
  WR = 0;  
  rc = 0;

# Default values of parameters

  meth =  3;
  nb   = -1;
  lc   =  1;

# Process parameters

  for k = 1 : 2 : length(args)
      switch args{k}
      case "method"
         switch args{k + 1};
         case "enhanced"
            meth = 3;
         case "modal"
            meth = 2;
         case "direct"
            meth = 1;
         otherwise
            printf("*E* mfs_freqresp: unknown method %s\n", args{k + 1});
            rc = 1;
         end 
      case "nband"
         nb = args{k + 1};
         if (nb < 0)
            printf("*E* mfs_freqresp: nband must be positive\n");
            rc = 1;
         end
      case "loadcase"
         lc = args{k + 1};
         if (lc < 1)
            printf("*E* mfs_freqresp: loadcase must be positive\n");
            rc = 1;
         end
      otherwise
         printf("*E* mfs_freqresp: unknown parameter %s\n", args{k})
         rc = 1;
      end
  end

  if (rc) return; end

# Compute response

  if (meth > 1)
     if (nb < 0)
        nb = 5;
     end
     [resp, ES, rc] = mfs_mfreq(cmp, f, nb, lc, meth);
  else
     [resp, ES, WR, rc] = mfs_dfreq(cmp, f, nb, lc);
  end

  if (rc); return; end

# End

  f = resp.freq;

  if (! isfield(cmp, "freqresp"))
     cmp.freqresp.nfa = lc;
  else
     cmp.freqresp.nfa = max(lc, cmp.freqresp.nfa);
  end
  cmp.freqresp.resp{lc} = resp;

end
