function [nodes, rc] = mfs_new_nodes(msg, input_nodes, ndim)

# usage: [nodes, rc] = mfs_new_nodes(msg, input_nodes, ndim)
#
# Input  msg           File handle of message file
#        input_nodes   Structure array with input data of the nodal
#                      points
#        ndim          Dimension (2 or 3)
#
# Output nodes         Structure with nodal point data
#        rc            Return code: 0 means no erros
#                                   1 means errors
#
# The function builds a structure with all nodal point data from the
# input data.
#
# Input data:
#
#    input_nodes(:).id          Nodal point identifier
#    input_nodes(:).coor(ndim)  Nodal point coordinates
#
# Output data:
#
#    nodes.nofnod               Number of nodal points
#    nodes.ncoor                Number of coordinates per nodal point
#    nodes.maxcor(ndim)         Maximum values of coordinates
#    nodes.mincor(ndim)         Minimum values of coordinates
#    nodes.ids(:)               Nodal point identifiers in ascending order
#    nodes.coor(:, ndim)        Nodal point coordinates 
#
# ---------------------------------------------------------------------

  node_fields = {"id", "coor"};

# Check arguments

  if (nargin != 3 || nargout != 2)
     print_usage();
  end

# Initialize

  rc    = 0;
  nodes = struct();

# Check input data

  if ((rc = mfs_checkfields1(msg, input_nodes, node_fields, 
                             "Nodal points")))
     return;
  end

# Get nodal point identifiers

  nofnod = length(input_nodes);
  nodids = [input_nodes.id];

  if (length(nodids) < nofnod)
     rc = 1;
     for n = 1 : nofnod
         if (isempty(input_nodes(n).id))
            fprintf(msg, "*E* Identifier of node number %6d undefined\n", n)
         end
     end
     return;
  end

  [nodids, index] = sort(nodids);

# Check on duplicate nodal point identifiers

  ixb = find(nodids(1 : nofnod-1) == nodids(2 : nofnod));
  if (! isempty(ixb))
     rc = 1;
     for ix = ixb
         fprintf(msg, "*E* Duplicate nodal point identifier %6d found\n",
                 nodids(ix));
     end
  end

# Get coordinates

  coor = zeros(nofnod, ndim);

  for n = 1 : nofnod
      ix = index(n);
      if (isvector(input_nodes(ix).coor))
         if (length(input_nodes(ix).coor) == ndim)
            coor(n, :) = input_nodes(ix).coor;
         else
            rc = 1;
            fprintf(msg, "*E* Wrong definition of coordinates of nodal point %6d\n",
                    input_nodes(ix).id);
         end
      else
         rc = 1;
         fprintf(msg, "*E* Coordinates of nodal point %6d undefined\n",
                input_nodes(ix).id);
      end
  end

  if (rc) return; end

  maxcor = max(coor);
  mincor = min(coor);

# Build the nodal point structure

  nodes = struct("nofnod", nofnod, "ncoor", ndim,
                 "maxcor", maxcor, "mincor", mincor,
                 "ids", nodids, "coor", coor);

end
