function [cmp, rc] = mfs_newx(msg, model, options)

# usage: [cmp, rc] = mfs_newx(msg, model, options)
#
# Input  msg     File handle of message file 
#        model   Structure with model data
#        options Structure with options
#
# Output cmp     Structure with component data
#                (0 if errors occurred)
#        rc      Return code: 0 = no errors
#                             1 = errors
#
# The function initializes a new component of type solid. The following
# fields are defined:
#
#  type         Component type
#  subtype      Component subtype
#  nodes        Structure with nodal point data
#  elements     Structure with element data
#  dofs         Structure with data on degrees of freedom
#  load         Structure with load data
#  nset         Structure with node sets
#  eset         Structure with element sets
#
# -----------------------------------------------------------------------------

  legal_fields = {"type", "subtype", "nodes", "elements", ...
                  "constraints", "loads", "nset", "eset", ...
                  "damping"};

  cmp = struct();

# Check arguments

  if (nargin != 3 || nargout != 2)
     print_usage();
  end

# Check fields

  if ((rc = mfs_checkfields4(msg, model, legal_fields, ...
                             "Model definition")))
     return;
  end

  rc  = 1;

# Model subtype

  if (isfield(model, "subtype"))
     switch model.subtype
     case "2d"
        ndim = 2;
     case "3d"
        ndim = 3;
     otherwise
        fprintf(msg, "*E* Unknown model subtype %s\n", model.subtype);
        return;
     end
  else
     fprintf(msg, "*E* Model subtype not defined\n");
     return;
  end

  fprintf(msg, "\n  Model Type = %s, Model Subtype = %s\n\n", ...
          model.type, model.subtype);

# Nodal points

  if (isfield(model, "nodes"))
     [nodes, rc] = mfs_new_nodes(msg, model.nodes, ndim);
  else
     fprintf(msg, "*E* Definition of nodal points missing\n");
  end

  if (rc) return; end

# Node sets

  if (isfield(model, "nset"))
     [nset, rc] = mfs_new_nset(msg, model.nset, nodes);
     if (rc) return; end
  else
     nset = struct();
  end

# Linear constraints

  if((isconst = isfield(model, "constraints")));
     constraints = model.constraints;
     [lincon, mxdofpnt, rcc] = ...
        mfs_new_lincon(msg, model.subtype, constraints, nodes);
     islincon = isstruct(lincon);
  else
     rcc         = 0;
     mxdofpnt    = 0;
     islincon    = 0;
     constraints = 0;
     lincon      = 0;
  end

# Elements

  if (isfield(model, "elements"))
     [elements, mxdofpnt, rce] = ...
        mfs_new_elements(msg, model.elements, nodes, model.subtype,
                         mxdofpnt);
     if (options.chkelts)
        if (! rce)
           rce = mfs_new_chkelts(msg, model.subtype, elements, nodes);
        end
     else
        printf("mfs_new: element checks switched off\n");
     end
  else
     fprintf(msg, "*E* Definition of elements missing\n");
     rce = 1;
  end

  rc = rcc || rce;
  if (rc) return; end

# Element sets

  if (isfield(model, "eset"))
     [eset, rc] = mfs_new_eset(msg, model.eset, 
                               elements.index);
     if (rc) return; end
  else
     eset = struct();
  end

# Degrees of freedom

  [dofs, rc] = mfs_new_dofs(msg, mxdofpnt, nodes, elements,
                            constraints, lincon);

  if (rc) return; end

# Loads

  if (isfield(model, "loads"))
     [load, rc] = mfs_new_load(msg, nodes, elements, dofs,
                               model.loads);
  else
     load = 0;
  end

# Damping

  if (isfield(model, "damping"))
     [damping, rcd] = mfs_damping(msg, model.damping);
     rc = rc || rcd;
  else
     damping = 0;
  end

# In case of errors, return now

  if (rc) return; end

# Build the component

  cmp = struct("type", model.type, "subtype", model.subtype,
               "nodes", nodes, "elements", elements, "dofs", dofs,
               "nset", nset, "eset", eset);
  if (isstruct(load))
     cmp = setfield(cmp, "load", load);
  end
  if (isstruct(damping))
     cmp = setfield(cmp, "damping", damping);
  end

# Print messages

  fprintf(msg, "  Number of nodes      = %5d", cmp.nodes.nofnod);
  fprintf(msg, ",  Number of elements = %5d\n", cmp.elements.nofelt);
  fprintf(msg, "  Number of element types = %2d\n", cmp.elements.noftyp);
  fprintf(msg, "  Number of global     degrees of freedom = %5d\n",
          cmp.dofs.ndofg);
  fprintf(msg, "  Number of local      degrees of freedom = %5d\n",
          cmp.dofs.ndofl);
  fprintf(msg, "  Number of prescribed degrees of freedom = %5d\n",
          cmp.dofs.ndofp);
  fprintf(msg, "  Number of dependent  degrees of freedom = %5d\n",
          cmp.dofs.ndofd);
  if (isfield(cmp, "load"))
     fprintf(msg, "\n  Number of load cases = %2d\n",
             cmp.load.nofldc);
  end
  
end
