function fgh = mfs_plotx(cmp, params)

# usage: fgh = mfs_plotx(cmp, params)
#
# Input  cmp       Structure with component
#        params    Plot parameters
#
# Output fgh       Figure handle
#
# The function plots a solid component.
#
# Plot parameters:
#
#   Name     Values   Default  Meaning
#  ---------------------------------------------------------------
#   nodid     0 / 1      0     Plot node identifiers
#   eltid     0 / 1      0     Plot element identifiers
#   deform    0 / 1, 2   0     Deformed plot: 2 = undeformed and
#                              deformed
#   force     0 / 1, 2   0     Plot applied forces: 2 = force labels
#   reac      0 / 1, 2   0     Plot reaction forces: 2 = force labels
#   loadcase   lcs       1     Loadcase list
#   modes      lcs       0     List of normal modes
#   beamaxis   0 / 1     0     Plot beam z-axis
#   scald               0.1    Scale factor for deformed plot
#   scalf               0.1    Scale factor for load arrows
#   scalax              0.4    Scale factor for beam axis
#   figure               0     Handle of existing figure
#
# Parameters defining graphic properties:
#
#   Name          Default      Meaning
#  ---------------------------------------------------------------
#   position                   Screen position of the plot
#   paperposition              Paper position of the plot
#   outerposition              Axis outerposition property
#   visible        "off"       Axis visibility
#   fontsize        14         Fontsize
#   fontname         *         Fontname
#   view         [30, 60]      Elevation (measured from z-axis) and
#                              azimuth
#
# ---------------------------------------------------------------------

# Copyright (c) 2017 by Johannes Wandinger


  global pltparm;

# Check arguments

  if (nargin != 2)
     print_usage();
  end

  Fmax = 0;  % Initialize force maximum

# Default values of parameters

  pltparm = struct("nodid", 0, "eltid", 0, "deform", 0,
                   "force", 0, "reac", 0, "loadcase", 1,
                   "modes", 0, "beamaxis", 0,
                   "scald", 0.1, "scalf", 0.1,
                   "scalax", 0.4, "figure", 0,
                   "position", [100, 100, 1000, 707],
                   "paperposition", [0, 0, 14, 9],
                   "outerposition", [0.05 0.05 0.9 0.9],
                   "visible", "off", "fontsize", 14,
                   "fontname", "*", "view", [30, 60],
                   "color", "black");

# Look for global default values

  dfl = get(0, "default");
  if (isfield(dfl, "defaultaxesfontname"))
     pltparm.fontname = getfield(dfl, "defaultaxesfontname");
  end
  if (isfield(dfl, "defaultaxesfontsize"))
     pltparm.fontsize = getfield(dfl, "defaultaxesfontsize");
  end

  if (! isstruct(cmp))
     error("mfs_plot: first argument must be a structure\n");
  end

# Process parameters

  pnames = fieldnames(pltparm);
  np     = length(params);
  k      = 1;

  while (k <= np)
      param = params{k++};
      if (k <= np)
         if (sum(strcmp(param, pnames)))
            pltparm = setfield(pltparm, param, params{k++});
         else
            error("mfs_plot: unknown parameter %s\n", param);
         end
      else
         error("mfs_plot: incomplete parameter definition\n"); 
      end
  end

# Check data

  modes = pltparm.modes;

  if (modes)
     lcs = modes;
     if(! isfield(cmp, "modes"))
        error("mfs_plot: normal modes missing\n");
     end
     if(lcs(end) > cmp.modes.nofmod)
        error("mfs_plot: mode %d does not exist\n", lcs(end));
     end
  else
     lcs = pltparm.loadcase;
     if (pltparm.force)
        if(! isfield(cmp, "load"))
           error("mfs_plot: loads missing\n");
        end
        if(lcs(end) > cmp.load.nofldc)
           error("mfs_plot: load case %d does not exist\n",
                 lcs(end));
        end
     end
     if (pltparm.deform || pltparm.reac)
        if(! isfield(cmp, "statresp"))
           error("mfs_plot: static response missing\n");
        end
        if(lcs(end) > cmp.statresp.nofldc)
           error("mfs_plot: load case %d does not exist\n",
                 lcs(end));
        end
     end
  end

# Undeformed plots
# ----------------

  if (! pltparm.deform)

     for l = lcs

         plot_label = 1;

         if (pltparm.figure)
            fgh = pltparm.figure;
         else
            fgh = figure("position", pltparm.position,
                         "paperposition", pltparm.paperposition);
            axes("outerposition", pltparm.outerposition);
         end
         axis("equal");
         hax = gca();
         set(hax, "visible", pltparm.visible);
         set(hax, "defaultaxesfontname", pltparm.fontname);
         set(hax, "defaultaxesfontsize", pltparm.fontsize);

         if (strcmp(cmp.subtype, "3d"))
            azimuth = pltparm.view(2); elevation = 90 - pltparm.view(1);
            view(azimuth, elevation);
         end

         mfs_plot_elements(cmp.elements, cmp.type, cmp.subtype);

         if (pltparm.nodid == 1)
            mfs_plot_nodes(cmp.nodes, cmp.subtype);
         end

         if (pltparm.force)
            if (isfield(cmp, "load") && isfield(cmp.load, "point"))
               Fmax = mfs_plot_force(cmp.nodes, cmp.load.point(l),
                              cmp.subtype, "green", pltparm.force - 1,
                              Fmax);
               if (plot_label) 
                  label = sprintf("Loadcase %2d:", l);
                  mfs_plot_title(label, cmp.subtype);
                  plot_label = 0;
               end
            else
               printf("*W* mfs_plot: No applied forces found to plot\n");
            end
         end

         if (pltparm.reac)
            if (isfield(cmp, "statresp") && isfield(cmp.statresp, "reac"))
               Fmax = mfs_plot_force(cmp.nodes, cmp.statresp.reac(l),
                              cmp.subtype, "red", pltparm.reac - 1,
                              Fmax);
               if (plot_label)
                  label = sprintf("Loadcase %2d:", l);
                  mfs_plot_title(label, cmp.subtype);
                  plot_label = 0;
               end
            else
               printf("*W* mfs_plot: No reaction forces found to plot\n");
            end
         end

         fn = findall(hax, "-property", "fontname");  % Currently, these
         set(fn, "fontname", pltparm.fontname);       % properties are
         fs = findall(hax, "-property", "fontsize");  % not inherited
         set(fs, "fontsize", pltparm.fontsize);       % from axes

     end                      % Loadcase loop

  end

# Deformed plots
# --------------

  if (pltparm.deform)

     nodes    = cmp.nodes;
     mxdofpnt = cmp.dofs.mxdofpnt;

# Scale displacements

     if (modes)
        disp = mfs_plot_defscal(nodes, cmp.modes.disp,
                                mxdofpnt, pltparm.scald, lcs);
     else
        disp = mfs_plot_defscal(nodes, cmp.statresp.disp,
                                mxdofpnt, pltparm.scald, lcs);
     end

# Loadcase loop

     [mr, lc] = size(disp);
     elements = cmp.elements;

     for l = lcs

         if (pltparm.figure)
            fgh = pltparm.figure;
         else
            fgh = figure("position", pltparm.position,
                         "paperposition", pltparm.paperposition);
            axes("outerposition", pltparm.outerposition);
         end
         axis("equal");
         hax = gca();
         set(hax, "visible", pltparm.visible);
         set(hax, "defaultaxesfontname", pltparm.fontname);
         set(hax, "defaultaxesfontsize", pltparm.fontsize);

         if (strcmp(cmp.subtype, "3d"))
            azimuth = pltparm.view(2); elevation = 90 - pltparm.view(1);
            view(azimuth, elevation);
         end

         if (pltparm.deform == 2)
            parmsav       = pltparm;
            pltparm.nodid = 0;
            pltparm.eltid = 0;
            pltparm.color = "blue";
            mfs_plot_elements(cmp.elements, cmp.type, cmp.subtype);
            pltparm = parmsav;
         end

         nodes.coor = mfs_plot_noddef(cmp.nodes, disp(:, l), mxdofpnt);

         elements.elem = mfs_plot_elcor(cmp.elements, nodes);

         mfs_plot_elements(elements, cmp.type, cmp.subtype);

         if (pltparm.nodid)
            mfs_plot_nodes(nodes, cmp.subtype);
         end

         if (pltparm.force)
            Fmax = mfs_plot_force(nodes, cmp.load.point(l),
                                  cmp.subtype, "green", 
                                  pltparm.force - 1, Fmax);
         end

         if (pltparm.reac)
            Fmax = mfs_plot_force(nodes, cmp.statresp.reac(l),
                                  cmp.subtype, "red",
                                  pltparm.reac - 1, Fmax);
         end

         if (modes)
            label = sprintf("Mode %2d: %10.3f Hz",
                            l, cmp.modes.freq(l));
         else
            label = sprintf("Loadcase %2d:", l);
         end

         mfs_plot_title(label, cmp.subtype);

         fn = findall(hax, "-property", "fontname");  % Currently, these
         set(fn, "fontname", pltparm.fontname);       % properties are
         fs = findall(hax, "-property", "fontsize");  % not inherited
         set(fs, "fontsize", pltparm.fontsize);       % from axes

     end                      % Loadcase loop

  end

end
