function [cmp, rc] = mfs_resultsx(cmp, class, items)

# usage: [cmp, rc] = mfs_resultsx(cmp, class, items)
#
# Input  cmp        Structure with component
#        class      Result class
#        items{}    List of items
#
# Output cmp        Structure with component
#        rc         Return code: 0 = no errors
#                                1 = errors
#
# The function computes secondary results in solid mechanics.
#
# List of results:
#
#  class = statresp
#        element    Element results: stress, strain,
#                                    stress resultants
#
#  class = trim
#        element    Element results: stress, strain,
#                                    stress resultants
#
#  class = freqresp
#        element    Element results: stress, strain,
#                                    stress resultants
#
#  class = transresp
#        element    Element results: stress, strain,
#                                    stress resultants
#
# --------------------------------------------------------------------

# Check arguments

  if (nargin < 3 || nargout != 2)
     print_usage();
  end

# Initialize

  rc  = 1;

# Switch according to class and item

  switch class

  case "statresp"

     if (! isfield(cmp, "statresp"))
        printf("*W* mfs_results: There are no primary results ");
        printf("of class ""statresp""\n");
        return;
     end

     for r = 1 : length(items)
         switch items{r}
         case "element"
            cmp.statresp.elres = ...
              mfs_results_statresp_elem(cmp.subtype, cmp.elements, 
                                        cmp.statresp.disp);
         otherwise
            printf("*E* mfs_results: Unknown result item \"%s\"\n", items{r});
            return;
         end
     end

  case "trim"

     if (! isfield(cmp, "trim"))
        printf("*W* mfs_results: There are no primary results ");
        printf("of class ""trim""\n");
        return;
     end

     for r = 1 : length(items)
         switch items{r}
         case "element"
            cmp.trim.elres = ...
              mfs_results_statresp_elem(cmp.subtype, cmp.elements, 
                                        cmp.trim.disp);
         otherwise
            printf("*E* mfs_results: Unknown result item \"%s\"\n", items{r});
            return;
         end
     end

  case "freqresp"

     if (! isfield(cmp, "freqresp"))
        printf("*W* mfs_results: There are no primary results ");
        printf("of class ""freqresp""\n");
        return;
     end

     for n = 1 : cmp.freqresp.nfa

         rsp = cmp.freqresp.resp{n};

         for r = 1 : length(items)
            switch items{r}
            case "element"
               if (rsp.nback)
                  w = 2 * pi * rsp.freqback;
                  cmp.freqresp.resp{n}.elres = ...
                  mfs_results_freqresp_elem(cmp.subtype, cmp.elements, 
                                            rsp.U, w);
                  printf("*I* mfs_results: Element results computed ");
                  printf("for load case %3.0d\n", n);
               else
                  printf("*W* mfs_results: No element results computed ");
                  printf("for load case %3.0d\n", n);
                  printf("    Use mfs_back to perform backtransformation first\n");
               end
            otherwise
               printf("*E* mfs_results: Unknown result item \"%s\"\n", items{r});
               return;
            end
         end

     end % Loop over analyses

  case "transresp"

     if (! isfield(cmp, "transresp"))
        printf("*W* mfs_results: There are no primary results ");
        printf("of class ""transresp""\n");
        return;
     end

     for n = 1 : cmp.transresp.nta

         rsp = cmp.transresp.resp{n};

         for r = 1 : length(items)
            switch items{r}
            case "element"
               if (rsp.nback)
                  cmp.transresp.resp{n}.elres = ...
                  mfs_results_transresp_elem(cmp.subtype, cmp.elements, 
                                            rsp.u, rsp.v);
                  printf("*I* mfs_results: Element results computed ");
                  printf("for load case %3.0d\n", n);
               else
                  printf("*W* mfs_results: No element results computed ");
                  printf("for load case %3.0d\n", n);
                  printf("    Use mfs_back to perform backtransformation first\n");
               end
            otherwise
               printf("*E* mfs_results: Unknown result item \"%s\"\n", items{r});
               return;
            end
         end

     end % Loop over analyses

  otherwise

     printf("*E* mfs_results: Unknown result class \"%s\"\n", class);
     return;

  end % class

  rc = 0;

end
