function [geom, e, rc] = mfs_I_section(args)

# usage: [geom, e, rc] = mfs_I_section(args)
#
# Input  args{1 : 4}  Width b, height h, thicknesses t and s
#        args{5}      "rot" (optional)
#
# Output geom         Structure with geometry data
#        e(2)         ey and ez of center of mass
#        rc           Return code: <> 0 in case of errors
#
# The function computes the cross section properties of an I section.
#
# -------------------------------------------------------------------------

# Copyright(c) by 2022 Johannes Wandinger

# Initialize

  geom = struct();
  e    = [0, 0];
  rc   = 1;
  rot  = 0;

# Check arguments

  lna = length(args);
  if (lna < 4 || lna > 5)
     printf("*E* mfs_beamsection: wrong number of parameters for ");
     printf("I-section\n");
     return;
  endif

  if (lna == 5)
     if (strcmp(args{5}, "rot"))
        rot = 1;
     else
        printf("*E* mfs_beamsection: wrong parameter \"%s\" ", args{4});
        printf("for I-section\n");
     endif
  endif

  b = args{1}; h = args{2}; t = args{3}; s = args{4};

  if (b <= 0 || h <= 0 || t <= 0 || s <= 0)
     printf("*E* mfs_beamsection: bad input data for I-section: ");
     printf("b = %g, h = %g, t = %g, s = %g\n", b, h, t, s);
     return;
  endif

# Compute cross section properties

  rc = 0;

  hr = h - 2 * t;
  geom.A   = 2 * b * t + hr * s;
  geom.Iy  = t^3 * b / 6 + 0.5 * (h - t)^2 * b * t + hr^3 * s / 12;
  geom.Iz  = b^3 * t / 6 + s^3 * hr / 12;
  geom.Iyz = 0;
  geom.IT  = 1.3 * (2 * b * t^3 + hr * s^3) / 3;
  e        = [0, 0];

  if (rot)
     Iy      = geom.Iy;
     geom.Iy = geom.Iz;
     geom.Iz = Iy;
  endif

endfunction
