function [geom, e, rc] = mfs_T_section(args)

# usage: [geom, e, rc] = mfs_T_section(args)
#
# Input  args{1 : 3}  Width b, height h, thickness t
#        args{4}      "rot" (optional)
#
# Output geom         Structure with geometry data
#        e(2)         ey and ez of center of mass
#        rc           Return code: <> 0 in case of errors
#
# The function computes the cross section properties of a T section.
#
# -------------------------------------------------------------------------

# Copyright(c) by 2022 Johannes Wandinger

# Initialize

  geom = struct();
  e    = [0, 0];
  rc   = 1;
  rot  = 0;

# Check arguments

  lna = length(args);
  if (lna < 3 || lna > 4)
     printf("*E* mfs_beamsection: wrong number of parameters for ");
     printf("T-section\n");
     return;
  endif

  if (lna == 4)
     if (strcmp(args{4}, "rot"))
        rot = 1;
     else
        printf("*E* mfs_beamsection: wrong parameter \"%s\" ", args{4});
        printf("for T-section\n");
     endif
  endif

  b = args{1}; h = args{2}; t = args{3};

  if (b <= 0 || h <= 0 || t <= 0)
     printf("*E* mfs_beamsection: bad input data for T-section: ");
     printf("b = %g, h = %g, t = %g\n", b, h, t);
     return;
  endif

# Compute cross section properties

  rc = 0;

  hr = h - t;
  Ab = b * t;
  Ah = hr * t;
  ey = (0.5 * t * Ab  + (t + 0.5 * hr) * Ah) / (Ab + Ah);
  geom.A   = Ab + Ah;
  geom.Iy  =  (t^2 / 12 + (ey - 0.5 * t)^2) * Ab;
  geom.Iy  +=  (hr^2 / 12 + (t + 0.5 * hr - ey)^2) * Ah;
  geom.Iz  = (b^3 * t  + hr * t^3) / 12;
  geom.Iyz = 0;
  geom.IT  = 1.12 * (Ab + Ah) * t^2 / 3;
  geom.CE  = [0, -ey + 0.5 * t];
  e        = [ey, 0];

  if (rot)
     Iy      = geom.Iy;
     geom.Iy = geom.Iz;
     geom.Iz = Iy;
     geom.CE = flip(geom.CE);
     e       = [0, ey];
  endif

endfunction
