function [geom, e, rc] = mfs_Z_section(args)

# usage: [geom, e, rc] = mfs_Z_section(args)
#
# Input  args{1}      "right" or "left"
#        args{2 : 5}  Width b, height h, thicknesses t and s
#        args{6}      "rot" (optional)
#
# Output geom         Structure with geometry data
#        e(2)         ey and ez of center of mass
#        rc           Return code: <> 0 in case of errors
#
# The function computes the cross section properties of a Z section.
#
# -------------------------------------------------------------------------

# Copyright(c) by 2022 Johannes Wandinger

  sgnyz = [1, -1];  % Sign of Iyz for left and right Z-section

# Initialize

  geom = struct();
  e    = [0, 0];
  rc   = 1;
  rot  = 0;

# Check arguments

  lna = length(args);
  if (lna < 5 || lna > 6)
     printf("*E* mfs_beamsection: wrong number of parameters for ");
     printf("Z-section\n");
     return;
  endif

  type = find(strcmp(args{1}, {"left", "right"}));
  if (! type)
     printf("*E* mfs_beamsection: wrong type \"%s\" for Z-section\n",
            type);
  endif

  if (lna == 6)
     if (strcmp(args{6}, "rot"))
        rot = 1;
     else
        printf("*E* mfs_beamsection: wrong parameter \"%s\" ", args{4});
        printf("for Z-section\n");
     endif
  endif

  b = args{2}; h = args{3}; t = args{4}; s = args{5};

  if (b <= 0 || h <= 0 || t <= 0 || s <= 0)
     printf("*E* mfs_beamsection: bad input data for Z-section: ");
     printf("b = %g, h = %g, t = %g, s = %g\n", b, h, t, s);
     return;
  endif

# Compute cross section properties

  rc = 0;

  bf = 0.5 *(b + s);
  hs = h - 2 * t;
  Af = bf * t;
  geom.A   = 2 * Af + s * hs;
  geom.Iy  = bf * t^3 / 6 + hs^3 * s / 12;
  geom.Iy  += 0.5 * (h - t)^2 * Af;
  geom.Iz  = bf^3 * t / 6 + hs * s^3 / 12;
  geom.Iz  += 0.5 * (bf - s)^2 * Af;
  geom.Iyz = sgnyz(type) * 0.5 * (h - t) * (bf - s) * Af;
  geom.IT  = 1.12 * (b * t^3 + (h - t) * s^3) / 3;
  e        = [0, 0];

  if (rot)
     Iy       = geom.Iy;
     geom.Iy  = geom.Iz;
     geom.Iz  = Iy;
     geom.Iyz = -geom.Iyz;
  endif

endfunction
