function [geom, e, rc] = mfs_box_section(args)

# usage: [geom, e, rc] = mfs_box_section(args)
#
# Input  args{1}      "thin" or "thick"
#        args{2 : 4}  Width b, height h, thickness t
#
# Output geom         Structure with geometry data
#        e(2)         ey and ez of center of mass
#        rc           Return code: <> 0 in case of errors
#
# The function computes the cross section properties of a box section.
#
# -------------------------------------------------------------------------

# Copyright(c) by 2022 Johannes Wandinger

# Initialize

  geom = struct();
  e    = [0, 0];
  rc   = 1;

# Check arguments

  if (length(args) != 4)
     printf("*E* mfs_beamsection: wrong number of parameters for ");
     printf("box-section\n");
     return;
  endif

  stype = args{1};

  if (! ismember(stype, {"thin", "thick"}))
     printf("mfs_beamsection: unknown type \"%s\" ", stype)
     printf("of box-section\n");
     return;
  endif

  b = args{2}; h = args{3}; t = args{4};

  if (b <= 0 || h <= 0 || t <= 0)
     printf("*E* mfs_beamsection: bad input data for box-section: ");
     printf("b = %g, h = %g, t = %g\n", b, h, t);
     return;
  endif

# Compute cross section properties

  rc = 0;

  switch stype

  case "thin" % Bredt's formulae
     geom.A   = 2 * t * (b + h);
     geom.Iy  = t * h^2 * (h + 3 * b) / 6;
     geom.Iz  = t * b^2 * (b + 3 * h) / 6;
     geom.Iyz = 0;
     geom.IT  = 2 * b^2 * h^2 * t / (b + h);

  case "thick" % cf. Dubbel, Taschenbuch für den Maschinenbau
     ba = b; ha = h;
     bi = ba - 2 * t;
     hi = ha - 2 * t;
     bm = ba - t;
     hm = ha - t;
     Aa = ba * ha;
     Ai = bi * hi;
     geom.A   = Aa - Ai;
     geom.Iy  = (ba * ha^3 - bi * hi^3) / 12;
     geom.Iz  = (ba^3 * ha - bi^3 * hi) / 12;
     geom.Iyz = 0;
     Am = bm * hm;
     Um = bm + hm;  % Half of circumference
     geom.IT  = (Aa + Ai) * t * Am / Um;

  endswitch

endfunction
