function [geom, e, rc] = mfs_ring_section(args)

# usage: [geom, e, rc] = mfs_ring_section(args)
#
# Input  args{1}      "thin" or "thick"
#        args{2 : 3}  thin section: radius r and thicknes t
#                     thick section: outer radius ra and inner radius ri
#
# Output geom         Structure with geometry data
#        e(2)         ey and ez of center of mass
#        rc           Return code: <> 0 in case of errors
#
# The function computes the cross section properties of a ring section.
#
# -------------------------------------------------------------------------

# Copyright(c) by 2022 Johannes Wandinger

# Initialize

  geom = struct();
  e    = [0, 0];
  rc   = 1;

# Check arguments

  if (length(args) != 3)
     printf("*E* mfs_beamsection: wrong number of parameters for ");
     printf("ring-section\n");
     return;
  endif

  stype = args{1};

  if (! ismember(stype, {"thin", "thick"}))
     printf("mfs_beamsection: unknown type \"%s\" ", stype)
     printf("of ring-section\n");
     return;
  endif

  a1 = args{2}; a2 = args{3};

  if (a1 <= 0 || a2 <= 0)
     printf("*E* mfs_beamsection: bad input data for ring-section: ");
     printf("r/ra = %g, t/ri = %g\n", a1, a2);
     return;
  endif

# Compute cross section properties

  rc = 0;

  switch stype

  case "thin"
     r = a1; t = a2;
     geom.A   = 2 * pi * r * t;
     geom.Iy  = pi * r^3 * t;
     geom.Iz  = geom.Iy;
     geom.Iyz = 0;
     geom.IT  = 2 * geom.Iy;

  case "thick"
     ra = a1; ri = a2;
     geom.A   = pi * (ra^2 - ri^2);
     geom.Iy  = 0.25 * (ra^2 + ri^2) * geom.A;
     geom.Iz  = geom.Iy;
     geom.Iyz = 0;
     geom.IT  = 2 * geom.Iy;

  endswitch

endfunction
