function [x, t] = mfs_freq2timex(X, df, dt, bc)

# usage: [x, t] = mfs_freq2timex(X, df, dt, bc)
#
# Input  X(:, :)  Array with Fourier transform
#                 (columns correspond to frequencies)
#        df       Frequency increment
#        dt       Time step (optional)
#        bc       Base correction (optional):
#                  0 = no correction (default)
#                  1 = subtract a constant
#                  2 = subtact a linear function
#
# Output x(:, :)  Array with time series
#                 (columns correspond to time steps)
#        t(:)     Array with time steps (optional)
#
# The function computes the inverse Fourier transform from the
# values of the Fourier transform at the positive frequencies.
#
# The base correction can be used to enforce the starting value to be
# zero and to remove a linear shift.
#
# ---------------------------------------------------------------------

# Copyright (c) 2021 by Johannes Wandinger

# Initialize

  x = []; t = [];

# Get time step and number of time steps

  T  = 1 / df;
  nX = columns(X);

  if (isempty(dt))
     nx = 2 * nX - 1;
     dt = T / nx;
  else
     nx = round(T / dt);
  end

# Perform inverse Fourier transform: The factor 2 compensates the 
# missing complex conjugate values, but the first coefficient does
# not have a complex conjugate partner.

  X(:, 2 : end) *= 2;
  x = real(ifft(X, nx, 2)) / dt;
  t = (0 : nx - 1) * dt;

# Base correction

  if (bc)
     x -= x(:, 1);
     if (bc == 2)
        f = (0 : nX - 1) * df;
        s = sum(real(2 * pi * i * f .* X), 2) / T;
        x -= s * t;
     end
  end

end
